﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Windows.Forms;
using ESRI.ArcGIS.ArcMapUI;
using ESRI.ArcGIS.Carto;
using ESRI.ArcGIS.Editor;
using ESRI.ArcGIS.esriSystem;
using ESRI.ArcGIS.Geodatabase;
using ESRI.ArcGIS.Geometry;

namespace MergeRules
{
    public class MergeRules : ESRI.ArcGIS.Desktop.AddIns.Button
    {
        IEditor m_pEditor;
        IEnumFeature m_pEnumFeature;
        IFeatureClass pFeatcls;
        double lGTotalVal;
        int lSCode;


        public MergeRules()
        {
            try
            {
                m_pEditor = (IEditor)ArcMap.Application.FindExtensionByName("esriEditor.Editor");
            }
            catch(Exception ex)
            {
                MessageBox.Show(ex.Message,"エラー",MessageBoxButtons.OK,MessageBoxIcon.Error);
                this.Enabled = false;
            }
        }


        protected override void OnClick()
        {
            try
            {
                // code to merge the features, evaluate the merge rules and assign values to fields appropriatly

                // start edit operation
                m_pEditor.StartOperation();
                m_pEnumFeature.Reset();

                // create a new feature to be the merge feature
                IFeature pCurFeature;
                IFeature pNewFeature;
                int lCount;
                pNewFeature = pFeatcls.CreateFeature();

                // create the new geometry.
                IGeometry pGeom;

                // initialize the default values for the new feature
                IRowSubtypes pOutRSType = (IRowSubtypes)pNewFeature;

                if (lSCode != 0)
                {
                    pOutRSType.SubtypeCode = lSCode;
                }
                pOutRSType.InitDefaultValues();

                // get the first feature
                pCurFeature = m_pEnumFeature.Next();
                IFields pFlds = pFeatcls.Fields;

                lCount = 1;
                IGeometryCollection pGeometryCollection = new GeometryBagClass();

                pGeom = pCurFeature.ShapeCopy;

                while (pCurFeature != null)
                {
                    // get the geometry
                    pGeometryCollection.AddGeometry(pCurFeature.ShapeCopy);

                    // now go through each field, if it has a domain associated with it, then
                    // evaluate the merge policy...
                    IField pFld;
                    IDomain pDomain;
                    ISubtypes pSubtypes = (ISubtypes)pFeatcls;

                    for (int i = 0; i <= pFlds.FieldCount - 1; i++)
                    {
                        pFld = pFlds.get_Field(i);
                        pDomain = pSubtypes.get_Domain(lSCode, pFld.Name);

                        if (pDomain != null)
                        {
                            if (pDomain.MergePolicy == esriMergePolicyType.esriMPTSumValues)    // Sum values
                            {
                                if (lCount == 1)
                                {
                                    pNewFeature.set_Value(i, pCurFeature.get_Value(i));
                                }
                                else
                                {
                                    pNewFeature.set_Value(i, double.Parse(pNewFeature.get_Value(i).ToString()) + double.Parse(pCurFeature.get_Value(i).ToString()));
                                }
                            }
                            else if (pDomain.MergePolicy == esriMergePolicyType.esriMPTAreaWeighted)    // Area/length weighted average
                            {
                                if (lCount == 1)
                                {
                                    pNewFeature.set_Value(i, double.Parse(pCurFeature.get_Value(i).ToString()) * (getGeomVal(pCurFeature) / lGTotalVal));
                                }
                                else
                                {
                                    pNewFeature.set_Value(i, double.Parse(pNewFeature.get_Value(i).ToString()) + (double.Parse(pCurFeature.get_Value(i).ToString()) * double.Parse(pCurFeature.get_Value(i).ToString()) * (getGeomVal(pCurFeature) / lGTotalVal)));
                                }
                            }
                        }
                        else
                        {
                            if (lCount == 1)
                            {
                                if (pFld.Editable == true)
                                {
                                    pNewFeature.set_Value(i, pCurFeature.get_Value(i));
                                }
                            }
                        }

                    }   //do not need a case for default value as it is set above



                    pCurFeature.Delete(); // delete the feature

                    pCurFeature = m_pEnumFeature.Next();
                    lCount += 1;

                }

                pGeom.SetEmpty();


                ITopologicalOperator pTopologicalOperator = (ITopologicalOperator)pGeom;
                pTopologicalOperator.ConstructUnion((IEnumGeometry)pGeometryCollection);

                //pNewFeature.Shape = pOutputGeometry
                pNewFeature.Shape = (IGeometry)pTopologicalOperator;
                pNewFeature.Store();

                // finish edit operation
                m_pEditor.StopOperation("Features merged");

                // refresh features
                IInvalidArea pRefresh = new InvalidAreaClass();
                pRefresh.Display = m_pEditor.Display;
                pRefresh.Add(pNewFeature);
                pRefresh.Invalidate(-2);    //esriAllScreenCaches



                // select new feature
                IMap pMap = m_pEditor.Map;
                pMap.ClearSelection();
                pMap.SelectFeature(FindLayer(pFeatcls, pMap), pNewFeature);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }


        protected override void OnUpdate()
        {
            try
            {
                // code to check if at least two features are selected,
                // and if so, if they are in the same f-class

                // First, check to see if an edit session is in progress
                if (m_pEditor.EditState == esriEditState.esriStateNotEditing)
                {
                    this.Enabled = false;
                    return;
                }

                // OK so we are editing. Is there more than 1 feature selected?
                if (m_pEditor.SelectionCount < 2)
                {
                    this.Enabled = false;
                    return;
                }

                // Multiple selected selected - are they from the same feature class,
                // and are they of the same subtype? Also, find the cumulative value
                // of their geometry sizes...
                m_pEnumFeature = m_pEditor.EditSelection;

                int lCLSID;
                IFeature pChkFeature;
                IRowSubtypes pRowSubtypes;

                pChkFeature = m_pEnumFeature.Next();


                // is this not a point class
                pFeatcls = (IFeatureClass)pChkFeature.Class;
                if (pFeatcls.ShapeType == esriGeometryType.esriGeometryMultipoint | pFeatcls.ShapeType == esriGeometryType.esriGeometryPoint)
                {
                    this.Enabled = false;
                    return;
                }



                // check features for class id, subtype code
                lCLSID = pChkFeature.Class.ObjectClassID;
                lGTotalVal = 0;
                pRowSubtypes = (IRowSubtypes)pChkFeature;
                lSCode = pRowSubtypes.SubtypeCode;



                while (pChkFeature != null)
                {
                    pRowSubtypes = (IRowSubtypes)pChkFeature;
                    if (pChkFeature.Class.ObjectClassID == lCLSID && pRowSubtypes.SubtypeCode == lSCode)
                    {
                        lGTotalVal = lGTotalVal + getGeomVal(pChkFeature);
                    }
                    else
                    {
                        //マージポリシーがないフィーチャ選択時はボタンを無効化
                        this.Enabled = false;
                        return;
                    }

                    pChkFeature = m_pEnumFeature.Next();
                }

                

                this.Enabled = true;
            }
            catch
            {
                //マージポリシーがないフィーチャ選択時はボタンを無効化
                this.Enabled = false;
            }

        }


        private ILayer FindLayer(IFeatureClass pFcls, IMap pMap)
        {
            ILayer functionReturnValue = default(ILayer);
            // helper function to find a layer for a feature class
            int i = 0;
            ILayer pLayer = default(ILayer);
            IFeatureLayer pFeatLayer = default(IFeatureLayer);

            for (i = 0; i <= pMap.LayerCount - 1; i++)
            {
                pFeatLayer = (IFeatureLayer)pMap.get_Layer(i);
                if (pFeatLayer.FeatureClass.ObjectClassID == pFcls.ObjectClassID)
                {
                    pLayer = pFeatLayer;
                    functionReturnValue = pLayer;
                }
            }
            return functionReturnValue;
        }


        private double getGeomVal(IFeature pFeature)
        {
            double functionReturnValue = 0;
            // helper function to get the area/length/perimeter of a feature
            IFeatureClass pFC = default(IFeatureClass);
            pFC = (IFeatureClass)pFeature.Class;
            IFields pvFlds = default(IFields);
            pvFlds = pFC.Fields;

            if (pFC.ShapeType == esriGeometryType.esriGeometryMultipoint | pFC.ShapeType == esriGeometryType.esriGeometryMultipoint | pFC.ShapeType == esriGeometryType.esriGeometryNull)
            {
                functionReturnValue = 0;
            }
            else if (pFC.ShapeType == esriGeometryType.esriGeometryPolygon)
            {
                functionReturnValue = (double)pFeature.get_Value(pvFlds.FindField(pFC.AreaField.Name));
            }
            else
            {
                functionReturnValue = (double)pFeature.get_Value(pvFlds.FindField(pFC.LengthField.Name));
            }
            return functionReturnValue;
        }
    }
}
