#coding:utf-8
#-------------------------------------------------------------------------------
# Name          : CreateStandardGridSquare
# Purpose:      : 標準地域メッシュ ポリゴン作成ツール (ArcGIS Pro 版)
#
# Created       : 2013年10月7日
# Copyright     : ESRIジャパン株式会社
# Disclaimer    : 本ツールはサンプル プログラムでありすべての動作を保証するものではありません。
#                 ESRIジャパン株式会社は本ツールによるいかなる損害も補償しません。
#-------------------------------------------------------------------------------
import math
import tempfile
import arcpy

#関数部
#-------------------------------------------------------------------------------
#メッシュ作成用諸元
# MemoryFCName  : 出力メモリ フィーチャクラス名
# Title         : メッシュ種類
# Width         : メッシュ幅（十進経緯度）
# Height        : メッシュ高さ（十進経緯度）
# Multiples     : 1次メッシュの範囲に対する倍数
class Grid:
    def __init__(self, level):
        self.MemoryFCName = r"in_memory\grid" + str(level)

        if level == 1:
            self.Title = "第1次メッシュ"
            self.Width = 1
            self.Height = 2.0 / 3.0
            self.Multiples = 1
        elif level == 2:
            self.Title = "第2次メッシュ"
            self.Width = 1.0 / 8.0
            self.Height = 2.0 / 3.0 / 8.0
            self.Multiples = 1 * 8
        elif level == 3:
            self.Title = "第3次メッシュ"
            self.Width = 1.0 / 8.0 / 10.0
            self.Height = 2.0 / 3.0 / 8.0 / 10.0
            self.Multiples = 1 * 8 * 10
        elif level == 4:
            self.Title = "第4次メッシュ"
            self.Width = 1.0 / 8.0 / 10.0 / 2.0
            self.Height = 2.0 / 3.0 / 8.0 / 10.0 / 2.0
            self.Multiples = 1 * 8 * 10 * 2
        elif level == 5:
            self.Title = "第5次メッシュ"
            self.Width = 1.0 / 8.0 / 10.0 / 2.0 / 2.0
            self.Height = 2.0 / 3.0 / 8.0 / 10.0 / 2.0 / 2.0
            self.Multiples = 1 * 8 * 10 * 2 * 2
        elif level == 6:
            self.Title = "第6次メッシュ"
            self.Width = 1.0 / 8.0 / 10.0 / 2.0 / 2.0 / 2.0
            self.Height = 2.0 / 3.0 / 8.0 / 10.0 / 2.0 / 2.0 / 2.0
            self.Multiples = 1 * 8 * 10 * 2 * 2
        elif level == 7:
            self.Title = "100mメッシュ（第3次メッシュ10分の1細分区画）"
            self.Width = 1.0 / 8.0 / 10.0 / 10.0
            self.Height = 2.0 / 3.0 / 8.0 / 10.0 / 10.0
            self.Multiples = 1 * 8 * 10 * 10
        elif level == 8:
            self.Title = "50mメッシュ（第3次メッシュ20分の1細分区画）"
            self.Width = 1.0 / 8.0 / 10.0 / 20.0
            self.Height = 2.0 / 3.0 / 8.0 / 10.0 / 20.0
            self.Multiples = 1 * 8 * 10 * 20

#-------------------------------------------------------------------------------
#メッシュ コードの作成
#引数
# level         : メッシュ レベル
# lon           : 十進経度
# lat           : 十進緯度
#戻り値：辞書型{}
# grid_code     : メッシュ番号
# grid_code2    : メッシュ番号（ハイフン表記）
# lon           : 南端緯度
# lat           : 南端経度
def getGridCode(level, lon, lat):
    #第1次メッシュ
    if level == 1:
        left_operator  = int(math.floor(lat * 15.0 / 10.0))  #緯度方向0.667(2/3)度区切り
        right_operator = int(math.floor(lon - 100))          #経度方向1度区切り
        dest_lon = right_operator + 100.0
        dest_lat = left_operator * 2.0 / 3.0

        src = {"grid_code":str(left_operator)+str(right_operator),"grid_code2":str(left_operator)+str(right_operator), "lon":dest_lon, "lat":dest_lat}

    #第2次メッシュ
    elif level == 2:
        base_data = getGridCode(1, lon, lat)

        left_operator = int(math.floor((lat - base_data["lat"]) * 100000.0 / 8333)) #緯度方向5分(5/60=0.08333)区切り
        right_operator = int(math.floor((lon - base_data["lon"]) * 1000.0 / 125.0)) #経度方向7分30秒(7/60+30/60/60=0.11666+0.008333=0.1249))区切り
        dest_lon = right_operator * 125.0 / 1000.0   + base_data["lon"]
        dest_lat = left_operator * 2.0 / 3.0 / 8.0 + base_data["lat"]

        src = {"grid_code":base_data["grid_code"]+str(left_operator)+str(right_operator),"grid_code2":base_data["grid_code"]+"-"+str(left_operator)+str(right_operator), "lon":dest_lon, "lat":dest_lat}

    #第3次メッシュ
    elif level == 3:
        base_data = getGridCode(2, lon, lat)

        left_operator = int(math.floor((lat - base_data["lat"]) * 1000000.0 / 8333))    #緯度方向30秒(30/60/60=0.008333)区切り
        right_operator = int(math.floor((lon - base_data["lon"]) * 10000.0 / 125.0))    #経度方向45秒(45/60/60=0.0125)区切り
        dest_lon = right_operator * 125.0 / 10000 + base_data["lon"]
        dest_lat = left_operator * 2.0/3.0/8.0/10.0 + base_data["lat"]

        src = {"grid_code":base_data["grid_code"]+str(left_operator)+str(right_operator),"grid_code2":base_data["grid_code2"]+"-"+str(left_operator)+str(right_operator), "lon":dest_lon, "lat":dest_lat}

    #第4次メッシュ
    elif level == 4:
        base_data = getGridCode(3, lon, lat)

        left_operator = int(math.floor((lat - base_data["lat"]) * 1000000.0 / 8333.0 * 2.0))    #緯度方向15秒(30/60/60/2=0.0041666)区切り
        right_operator = int(math.floor((lon - base_data["lon"]) * 10000.0 / 125.0 * 2.0))      #経度方向22.5秒(45/60/60/2=0.00625)区切り
        dest_lon = right_operator * 125.0 / 10000 / 2 + base_data["lon"]
        dest_lat = left_operator * 2.0/3.0/8.0/10.0/2.0 + base_data["lat"]

        src = {"grid_code":base_data["grid_code"]+str(getSubCode(left_operator, right_operator)),"grid_code2":base_data["grid_code2"]+"-"+str(getSubCode(left_operator, right_operator)), "lon":dest_lon, "lat":dest_lat}

    #第5次メッシュ
    elif level == 5:
        base_data = getGridCode(4, lon, lat)

        left_operator = int(math.floor((lat - base_data["lat"]) * 1000000.0 / 8333 * 2 * 2))    #緯度方向7.5秒(30/60/60/2/2=0.00208333)区切り
        right_operator = int(math.floor((lon - base_data["lon"]) * 10000.0 / 125 * 2 * 2))      #経度方向11.25秒(45/60/60/2/2=0.003125)区切り
        dest_lon = right_operator * 125.0 / 10000 / 2 / 2 + base_data["lon"]
        dest_lat = left_operator * 2.0/3.0/8.0/10.0/2.0/2.0 + base_data["lat"]

        src = {"grid_code":base_data["grid_code"]+str(getSubCode(left_operator, right_operator)),"grid_code2":base_data["grid_code2"]+"-"+str(getSubCode(left_operator, right_operator)), "lon":dest_lon, "lat":dest_lat}

    #第6次メッシュ
    elif level == 6:
        base_data = getGridCode(5, lon, lat)

        left_operator = int(math.floor((lat - base_data["lat"]) * 1000000 / 8333.0 * 2 * 2 * 2))    #緯度方向 3.75秒(30/60/60/2/2/2=0.0010416666)区切り
        right_operator = int(math.floor((lon - base_data["lon"]) * 10000 / 125.0 * 2 * 2 * 2))      #経度方向 5.625秒(45/60/60/2/2/2=0.0015625)区切り
        dest_lon = right_operator * 125.0 / 10000 / 2 / 2 / 2 + base_data["lon"]
        dest_lat = left_operator * 2.0/3.0/8.0/10.0/2.0/2.0/2.0 + base_data["lat"]

        src = {"grid_code":base_data["grid_code"]+str(getSubCode(left_operator, right_operator)),"grid_code2":base_data["grid_code2"]+"-"+str(getSubCode(left_operator, right_operator)), "lon":dest_lon, "lat":dest_lat}

    #3次メッシュの10分の1（100mメッシュ）
    elif level == 7:
        base_data = getGridCode(3, lon, lat) #3次メッシュから計算

        left_operator = int(math.floor((lat - base_data["lat"]) * 1000000 / 8333 * 10))     #緯度方向3秒(30/60/60/10=0.0008333)区切り（3次メッシュの1/10）
        right_operator = int(math.floor((lon - base_data["lon"]) * 10000 / 125 * 10))       #経度方向4.5秒(45/60/60/10=0.00125)区切り（3次メッシュの1/10）
        dest_lon = right_operator * 125.0 / 10000 / 10 + base_data["lon"]
        dest_lat = left_operator * 2.0 / 3.0 / 8.0 / 10.0 / 10.0 + base_data["lat"]

        src = {"grid_code":base_data["grid_code"],"grid_code2":base_data["grid_code2"], "lon":dest_lon, "lat":dest_lat}

    #3次メッシュの20分の1（500mメッシュ）
    elif level == 8:
        base_data = getGridCode(3, lon, lat) #3次メッシュから計算

        left_operator = int(math.floor((lat - base_data["lat"]) * 1000000.0 / 8333 * 20))   #緯度方向15秒(30/60/60/20=0.0041666)区切り（3次メッシュの1/20）
        right_operator = int(math.floor((lon - base_data["lon"]) * 10000.0 / 125 * 20))     #経度方向4.5秒(45/60/60/20=0.00625)区切り（3次メッシュの1/20）
        dest_lon = right_operator * 125.0 / 10000 / 20 + base_data["lon"]
        dest_lat = left_operator * 2.0 / 3.0 / 8.0 / 10.0 / 20.0 + base_data["lat"]

        src = {"grid_code":base_data["grid_code"],"grid_code2":base_data["grid_code2"], "lon":dest_lon, "lat":dest_lat}

    return src

#-------------------------------------------------------------------------------
#第4次～第6次メッシュコード計算用関数
#引数
# left_operator : 左側コード
# right_operator: 右側コード
#戻り値:
# int           : 第4次～第6次のコード番号
def getSubCode(left_operator, right_operator):
    if left_operator == 0 and right_operator == 0:
        return 1
    elif left_operator == 0 and right_operator == 1:
        return 2
    elif left_operator == 1 and right_operator == 0:
        return 3
    elif left_operator == 1 and right_operator == 1:
        return 4
    else:
        return None

#-------------------------------------------------------------------------------
#座標系判定
#引数
# SpatialReference  : 空間参照
#戻り値
# bool              : 地理座標系の場合は真、投影座標系の場合は偽
def isGCS(SpatialReference):
    if "PROJCS" in SpatialReference.exportToString():
        return False
    else:
        return True

#-------------------------------------------------------------------------------
#ジオプロセシングの実行
#引数
# level             : メッシュ レベル
# min_lon           : メッシュの南端緯度
# min_lat           : メッシュの西端経度
# width             : 入力フィーチャクラスの幅
# height            : 入力フィーチャクラスの高さ
# FeatureLayer      : 入力フィーチャ レイヤ
# intersect_featuer : フィーチャをインターセクトするかどうか
# fieldName         : フィールド名
# sr                : 空間参照
#戻り値
# なし
def runGP(inGridName,level,min_lon,min_lat,width,height,FeatureLayer,intersect_feature,fieldName,sr):

    arcpy.SetProgressor("default",Grid(level).Title + "を作成しています...")

    gridcode = getGridCode(level, min_lon, min_lat) ##データの範囲の最小の矩形で原点を指定
    grid_string = str(gridcode["lon"]) + " " + str(gridcode["lat"])

    if arcpy.Exists(Grid(level).MemoryFCName):
        arcpy.Delete_management(Grid(level).MemoryFCName)

    memory_fc = Grid(level).MemoryFCName

    #メッシュ作成の行列数を計算
    number_rows = int(math.ceil(height / Grid(level).Height)) + 1
    number_columns = int(math.ceil(width / Grid(level).Width)) + 1
    result = arcpy.GridIndexFeatures_cartography(memory_fc, FeatureLayer, intersect_feature, "NO_USEPAGEUNIT", "", str(Grid(level).Width) +" DecimalDegrees", str(Grid(level).Height) + " DecimalDegrees", grid_string, str(number_rows), str(number_columns), "1", "NO_LABELFROMORIGIN")

    #フィールドの削除
    arcpy.DeleteField_management(memory_fc, ["PageName","PageNumber"])

    #フィールドの追加
    arcpy.AddField_management(memory_fc,fieldName[0] , "SHORT")
    arcpy.AddField_management(memory_fc,fieldName[1] , "TEXT")

    # UpdateCursor 関数を使用して Cursorオブジェクトを取得
    cursor = arcpy.da.UpdateCursor(memory_fc, ["SHAPE@XY",fieldName[0],fieldName[1]])

    for row in cursor:
        point = row[0]

        meshcode = getGridCode(level, point[0], point[1])
        row[1] = level
        row[2] = meshcode.get("grid_code")
        cursor.updateRow(row)

    #投影法の定義
    arcpy.DefineProjection_management(memory_fc,sr)

    #データのコピー
    fullPath = outWorkspace + "\\" + inGridName[level]
    result = arcpy.CopyFeatures_management(memory_fc, fullPath)
    fc = result.getOutput(0)

    del memory_fc,fc

#ジオプロセシング ツール部
#-------------------------------------------------------------------------------
#環境設定
arcpy.env.scratchWorkspace = tempfile.gettempdir()  #テンポラリ領域の取得
gcs_fc = arcpy.env.scratchGDB + r"\gcs_fc"          #スクラッチ ワークスペース用フィーチャクラス

#ジオプロセシングの引数
inFeatureLayer = arcpy.GetParameterAsText(0)        #入力フィーチャ レイヤ
inIntersected = arcpy.GetParameterAsText(1)         #インターセクトするかどうか
inGrids = arcpy.GetParameterAsText(2)               #作成メッシュ
outWorkspace = arcpy.GetParameterAsText(3)          #出力ワークスペース
srName = arcpy.GetParameterAsText(4)                #空間参照
unlimitedMode = arcpy.GetParameterAsText(5)         #制限解除
inGridName = ["dummy"]
inGridName.append(arcpy.GetParameterAsText(6))      #第1次メッシュファイル名
inGridName.append(arcpy.GetParameterAsText(7))      #第2次メッシュファイル名
inGridName.append(arcpy.GetParameterAsText(8))      #第3次メッシュファイル名
inGridName.append(arcpy.GetParameterAsText(9))      #第4次メッシュファイル名
inGridName.append(arcpy.GetParameterAsText(10))     #第5次メッシュファイル名
inGridName.append(arcpy.GetParameterAsText(11))     #第6次メッシュファイル名
inGridName.append(arcpy.GetParameterAsText(12))     #第7次メッシュファイル名
inGridName.append(arcpy.GetParameterAsText(13))     #第8次メッシュファイル名

fieldName = ("GRID_LEVEL","GRID_CODE")              #フィールド名

try:
    #投影法の定義
    if srName == "GCS_JGD_2011":
        sr = arcpy.SpatialReference(104020) #GCS_JGD_2011
    elif srName == "GCS_Tokyo":
        sr = arcpy.SpatialReference(4301)   #GCS_Tokyo
    else:
        sr = arcpy.SpatialReference(4612)   #JGD_2000

    arcpy.env.outputCoordinateSystem = sr   #出力座標系の設定

    #入力フィーチャでインターセクトするかどうか
    if inIntersected == "false":
        intersect_feature = "INTERSECTFEATURE"
    else:
        intersect_feature = "NO_INTERSECTFEATURE"

    # ArcGIS Pro向けに、以下の箇所を修正
    #fl =arcpy.mapping.Layer(inFeatureLayer)
    fl = inFeatureLayer

    descFL = arcpy.Describe(fl)
    if hasattr(descFL, 'dataElement'):
        layerSR = descFL.dataElement.spatialReference   #レイヤの空間参照
    elif hasattr(descFL, 'spatialReference'):
        layerSR = descFL.spatialReference   #データセットの空間参照        

    #座標系の判定
    if isGCS(layerSR) == False:
        arcpy.SetProgressor("default","座標系を判定しています...")
        if arcpy.Exists(gcs_fc):
            arcpy.Delete_management(gcs_fc)
        arcpy.Project_management(fl,gcs_fc,sr)
        result = arcpy.MakeFeatureLayer_management(gcs_fc,"new_Layer")
        fl = result.getOutput (0)
        descFL = arcpy.Describe(fl)
        if hasattr(descFL, 'dataElement'):
            layerSR = descFL.dataElement.spatialReference #レイヤの空間参照
        elif hasattr(descFL, 'spatialReference'):
            layerSR = descFL.spatialReference   #データセットの空間参照   
        del gcs_fc

    #入力フィーチャと出力フィーチャクラスの座標系の競合検知
    if layerSR.name != sr.name:
        arcpy.SetProgressor("default","中断しました")
        arcpy.AddWarning("入フィーチャと出力フィーチャクラスの測地系が異なります。処理を中断しました。")
        sys.exit()

    # ArcGIS Pro向けに、以下の箇所を修正
    #extent = fl.getExtent()
    extent=descFL.extent
    inGridsList = inGrids.split(';')

    #リミッタ制御
    flugError = False
    extentJapan = arcpy.Extent(122,20,154,46)
    errorMessage = ""

    arcpy.SetProgressor("default","リミッタ制限を判定しています...")
    if unlimitedMode == "false":
        #第1次メッシュ作成
        if "第1次メッシュ" in inGridsList and extentJapan.contains(extent) == False:
            errorMessage = "第1次メッシュ "
            flugError = True
        #第2次メッシュの作成
        if "第2次メッシュ" in inGridsList and extentJapan.contains(extent) == False:
            errorMessage = errorMessage + "第2次メッシュ "
            flugError = True
        #第3次メッシュの作成
        if "第3次メッシュ" in inGridsList and (extent.width > 1.1 or extent.height > 0.7):
            errorMessage = errorMessage + "第3次メッシュ "
            flugError = True
        #第4次メッシュの作成
        if "第4次メッシュ（2分の1地域メッシュ）" in inGridsList and (extent.width > 1.01 or extent.height > 0.7):
            errorMessage = errorMessage + "第4次メッシュ "
            flugError = True
        #第5次メッシュの作成
        if "第5次メッシュ（4分の1地域メッシュ）" in inGridsList and (extent.width > 1.01 or extent.height > 0.7):
            errorMessage = errorMessage + "第5次メッシュ "
            flugError = True
        #第6次メッシュの作成
        if "第6次メッシュ（8分の1地域メッシュ）" in inGridsList and (extent.width > 1.01 or extent.height > 0.7):
            errorMessage = errorMessage + "第6次メッシュ "
            flugError = True
        #100mメッシュ（第3次メッシュ10分の1細分区画）
        if "100mメッシュ（第3次メッシュ10分の1細分区画）" in inGridsList and (extent.width > 1.01 / 8.0 or extent.height > 0.67 / 8.0):
            errorMessage = errorMessage + "100mメッシュ（第3次メッシュ10分の1細分区画） "
            flugError = True
        #50mメッシュ（第3次メッシュ20分の1細分区画）
        if "50mメッシュ（第3次メッシュ20分の1細分区画）" in inGridsList and (extent.width > 1.01 / 8.0 or extent.height > 0.67 / 8.0):
            errorMessage = errorMessage + "50mメッシュ（第3次メッシュ20分の1細分区画） "
            flugError = True

        if flugError == True:
            arcpy.SetProgressor("default","中断しました")
            errorMessage = errorMessage + "がリミッターの範囲を超えています。リミッターにより処理が中断されました。"
            arcpy.AddWarning(errorMessage)
            sys.exit()

    #第1次メッシュ作成
    if "第1次メッシュ" in inGridsList:
        runGP(inGridName,1,extent.XMin,extent.YMin,extent.width,extent.height,fl,intersect_feature,fieldName,sr)
    #第2次メッシュの作成
    if "第2次メッシュ" in inGridsList:
        runGP(inGridName,2,extent.XMin,extent.YMin,extent.width,extent.height,fl,intersect_feature,fieldName,sr)
    #第3次メッシュの作成
    if "第3次メッシュ" in inGridsList:
        runGP(inGridName,3,extent.XMin,extent.YMin,extent.width,extent.height,fl,intersect_feature,fieldName,sr)
    #第4次メッシュの作成
    if "第4次メッシュ（2分の1地域メッシュ）" in inGridsList:
        runGP(inGridName,4,extent.XMin,extent.YMin,extent.width,extent.height,fl,intersect_feature,fieldName,sr)
    #第5次メッシュの作成
    if "第5次メッシュ（4分の1地域メッシュ）" in inGridsList:
        runGP(inGridName,5,extent.XMin,extent.YMin,extent.width,extent.height,fl,intersect_feature,fieldName,sr)
    #第6次メッシュの作成
    if "第6次メッシュ（8分の1地域メッシュ）" in inGridsList:
        runGP(inGridName,6,extent.XMin,extent.YMin,extent.width,extent.height,fl,intersect_feature,fieldName,sr)
    #第7次メッシュの作成
    if "100mメッシュ（第3次メッシュ10分の1細分区画）" in inGridsList:
        runGP(inGridName,7,extent.XMin,extent.YMin,extent.width,extent.height,fl,intersect_feature,fieldName,sr)
    #第8次メッシュの作成
    if "50mメッシュ（第3次メッシュ20分の1細分区画）" in inGridsList:
        runGP(inGridName,8,extent.XMin,extent.YMin,extent.width,extent.height,fl,intersect_feature,fieldName,sr)

except arcpy.ExecuteError as e:
    arcpy.SetProgressor("default","中断しました")
    arcpy.AddError(str(e))
    print (str(e).decode("UTF-8"))
